#include "petiga.h"
#include<math.h>

typedef struct {
  PetscInt   nf, nr, np;
  PetscInt nReactions;
  PetscInt init;
  PetscReal *alpha; /*[nr]*/
  PetscReal  *beta; /*[np]*/
  PetscReal *theta_c; /*[nf][nf]*/
  PetscReal *cahn; /*[nf][nf]*/
  PetscReal *lsigma; /*[nf][nf]*/
  PetscReal *stoichiometry_matrix; /*[nf][2] be aware this will depend upon type of chemical reaction*/
  PetscReal *Diff;
  PetscReal l;
  PetscReal e;
  PetscReal  kappa;
  PetscReal  delta;
  PetscReal  zeta;
  PetscReal kplus, kminus;  
  PetscScalar phibar;
  PetscReal eps;
  PetscReal C;
  PetscReal M0;
  PetscReal theta;
  PetscReal L0,Sprev[3];
  PetscReal energy,converged;
  PetscScalar factor;
  PetscScalar *omega_phi;
  PetscScalar Jphi;
  IGAProbe Unit;
  IGAProbe UAC;
  PetscScalar Psi0;
} AppCtx;

#undef __FUNCT__
#define __FUNCT__ "FirstPiolaKirchkoff"
PetscErrorCode FirstPiolaKirchkoff(PetscInt dim, PetscScalar phi0[], PetscScalar phi[], PetscScalar det, PetscScalar F_[], PetscScalar InvF_[], PetscScalar P_[], void *ctx)
{

  AppCtx *user = (AppCtx *)ctx;
  
  PetscFunctionBegin;

  if(det < 0.0){
  SETERRQ(PETSC_COMM_WORLD,
          PETSC_ERR_ARG_OUTOFRANGE,
          "Jacobian is negative");
  }
  
  PetscScalar (*F)[dim]     = (PetscScalar (*)[dim])  (F_);
  PetscScalar (*InvF)[dim]     = (PetscScalar (*)[dim])  (InvF_);
  PetscScalar (*P)[dim]     = (PetscScalar (*)[dim])  (P_);
  
  PetscInt i,j;
  PetscScalar beta = user->factor;
  
  PetscScalar VolSweSum = 0.0;
  for (i=0;i<user->nf;i++){VolSweSum += user->omega_phi[i]*(phi[i]-phi0[i]);}

  user->Jphi = 1.0 + VolSweSum;
  
  if(user->Jphi< 0.0){
  SETERRQ(PETSC_COMM_WORLD,
          PETSC_ERR_ARG_OUTOFRANGE,
          "Volumetric Swelling negative [Fix parameters]");
  }

  PetscScalar Jphi_13 = pow(user->Jphi,((PetscScalar)-1/3));
  PetscScalar Jphi_det = pow(Jphi_13,((PetscScalar)dim));
  
  PetscScalar NegativeDetCheck = det*Jphi_det;
  
  if(NegativeDetCheck < 0.0){
  SETERRQ(PETSC_COMM_WORLD,
          PETSC_ERR_ARG_OUTOFRANGE,
          "Determinant of Fe is negative");
  }
  
  for(i=0;i<dim;i++){
    for(j=0;j<dim;j++){
      P[i][j] = Jphi_13*(Jphi_13*F[i][j] - pow((Jphi_det*det),-beta)*pow(Jphi_13,((PetscScalar)-1))*InvF[j][i]);
    }
  }
  
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "MatProd"
PetscErrorCode MatProd(PetscInt dim, PetscScalar A_[], PetscScalar B_[], PetscScalar C_[])
{
  PetscFunctionBegin;
  PetscInt i,j,k;
  PetscScalar (*A)[dim]        = (PetscScalar (*)[dim])       (A_);
  PetscScalar (*B)[dim]        = (PetscScalar (*)[dim])       (B_);
  PetscScalar (*C)[dim]        = (PetscScalar (*)[dim])       (C_);

  PetscScalar sum_row_col; 
  for (i=0;i<dim;i++){
    for (j=0;j<dim;j++){
      sum_row_col= 0.0;
      for (k=0;k<dim;k++){
	sum_row_col += A[i][k]*B[k][j];
      }
      C[i][j] = sum_row_col;
    }
  }
  
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "TensorOperations"
PetscErrorCode TensorOperations(PetscInt dim, PetscScalar *det, PetscScalar B_[], PetscScalar A_[], PetscScalar InvA_[], void *ctx)
{
  
  AppCtx *user = (AppCtx *)ctx;
  PetscErrorCode ierr;
   
  PetscFunctionBegin;

  if (dim==3){

    PetscScalar (*B)[dim] = (PetscScalar (*)[dim]) (B_);
    PetscScalar (*A)[dim] = (PetscScalar (*)[dim]) (A_);
    PetscScalar (*InvA)[dim] = (PetscScalar (*)[dim]) (InvA_);
    
    PetscInt i,j;
    PetscScalar trace_A,trace_A2;
    PetscScalar Iden[dim][dim];
    PetscScalar  A2[dim][dim];

    // Forming F = I + gradX_u;

    for (i=0;i<dim;i++){
      for (j=0;j<dim;j++){
	if(i==j){Iden[i][j] = 1.0;}
	else{Iden[i][j] = 0.0;}
	A[i][j] = Iden[i][j] + user->l*B[i][j];
      }
    }
    
    *det = 0.0;
    *det +=   A[0][0]*A[1][1]*A[2][2] + A[0][1]*A[1][2]*A[2][0] + A[0][2]*A[1][0]*A[2][1];
    *det += - A[0][2]*A[1][1]*A[2][0] - A[0][1]*A[1][0]*A[2][2] - A[0][0]*A[1][2]*A[2][1];
 
    ierr = MatProd(dim, &A[0][0], &A[0][0], &A2[0][0]);CHKERRQ(ierr);
    trace_A = 0.0; for (i=0;i<dim;i++){trace_A += A[i][i];} 
    trace_A2 = 0.0; for (i=0;i<dim;i++){trace_A2 += A2[i][i];}

    for(i=0; i<dim; i++){
      for(j=0; j<dim; j++){
	InvA[i][j] = (1/(*det))*(0.5*((trace_A)*(trace_A)-trace_A2)*Iden[i][j]-A[i][j]*trace_A + A2[i][j]);
      }
    }
  }
  
  if (dim==2){

    PetscScalar (*B)[dim] = (PetscScalar (*)[dim]) (B_);
    PetscScalar (*A)[dim] = (PetscScalar (*)[dim]) (A_);
    PetscScalar (*InvA)[dim] = (PetscScalar (*)[dim]) (InvA_);
        
    PetscInt i,j;
    PetscScalar trace_A;
    PetscScalar Iden[dim][dim];

    // Forming F = I + gradX_u;                                                                                                                                                      
    for (i=0;i<dim;i++){
      for (j=0;j<dim;j++){
        if(i==j){Iden[i][j] = 1.0;}
        else{Iden[i][j] = 0.0;}
        A[i][j] = Iden[i][j] + user->l*B[i][j];
      }
    }

    *det = 0.0;
    *det += A[0][0]*A[1][1] - A[0][1]*A[1][0];
         
    trace_A = 0.0; for (i=0;i<dim;i++){trace_A += A[i][i];}
    
    for(i=0; i<dim; i++){
      for(j=0; j<dim; j++){
        InvA[i][j] = (1/(*det))*(trace_A*Iden[i][j]-A[i][j]);
      }
    }
  }    
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "MaterialJacobianGradient"
PetscErrorCode MaterialJacobianGradient(PetscInt dim, PetscScalar F_[], PetscScalar Hess_[], PetscScalar J_grad[])
{
 PetscFunctionBegin;

 PetscScalar (*Hess)[dim][dim] = (PetscScalar (*)[dim][dim]) (Hess_);
 PetscScalar (*F)[dim] = (PetscScalar (*)[dim]) (F_);
 PetscInt i;
 
 if(dim==2){
   for(i=0;i<dim;i++){
     J_grad[i] = Hess[0][0][i]*F[1][1] + F[0][0]*Hess[1][1][i] - Hess[0][1][i]*F[1][0] - F[0][1]*Hess[1][0][i];
   }
 }

 if(dim==3){
   for(i=0;i<dim;i++){
     J_grad[i] = 0.0;
     J_grad[i] += Hess[0][0][i]*(F[1][1]*F[2][2]-F[2][1]*F[1][2]);
     J_grad[i] += F[0][0]*(Hess[1][1][i]*F[2][2] + F[1][1]*Hess[2][2][i] - Hess[2][1][i]*F[1][2] - F[2][1]*Hess[1][2][i]);
     J_grad[i] -= Hess[1][0][i]*(F[0][1]*F[2][2]-F[2][1]*F[0][2]);
     J_grad[i] -= F[1][0]*(Hess[0][1][i]*F[2][2] + F[0][1]*Hess[2][2][i] - Hess[2][1][i]*F[0][2] - F[2][1]*Hess[0][2][i]);
     J_grad[i] += Hess[2][0][i]*(F[0][1]*F[1][2]-F[1][1]*F[0][2]);
     J_grad[i] += F[2][0]*(Hess[0][1][i]*F[1][2] + F[0][1]*Hess[1][2][i] - Hess[1][1][i]*F[0][2] - F[1][1]*Hess[0][2][i]);
   }
 }
  
 PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "HessianInverse"
PetscErrorCode HessianInverse(PetscInt dim, PetscScalar FHess_[], PetscScalar InvF_[], PetscScalar InvHessP_[], PetscScalar InvHessR[])
{
  PetscFunctionBegin;

  PetscScalar (*InvF)[dim] = (PetscScalar (*)[dim]) (InvF_);
  PetscScalar (*FHess)[dim][dim] = (PetscScalar (*)[dim][dim]) (FHess_);
  PetscScalar (*InvHessP)[dim][dim] = (PetscScalar (*)[dim][dim]) (InvHessP_);
  PetscScalar Inter_sum;
  PetscInt i,j,k,l,m;

  if(InvHessP_){
    // Hassein tensor to be used in pressure function
    for(i=0;i<dim;i++){
      for(j=0;j<dim;j++){
	for(k=0;k<dim;k++){
	  Inter_sum = 0.0;
	  for(l=0;l<dim;l++){
	    for(m=0;m<dim;m++){
	      Inter_sum -= FHess[m][l][k]*InvF[l][j]*InvF[i][m];
	    }
	  }
	  InvHessP[i][j][k] = Inter_sum;
	}
      }
    }
  }

  if(InvHessR){
  // Vector to be used in Residual function
    for(k=0;k<dim;k++){
      Inter_sum = 0.0;
      for(i=0;i<dim;i++){
	for(j=0;j<dim;j++){
	  for(l=0;j<dim;j++){
	    for(m=0;j<dim;j++){
	      Inter_sum -= FHess[i][j][m]*InvF[m][l]*InvF[j][l]*InvF[k][i];
	    }
	  }
	}
      }
      InvHessR[k] = Inter_sum;
    }    
  }
  
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "MaterialPressureGradient"
PetscErrorCode MaterialPressureGradient(PetscInt dim, PetscScalar det, PetscScalar phi0[], PetscScalar phi[], PetscScalar dphi0_[], PetscScalar dphi_[],
					PetscScalar FHess_[], PetscScalar F_[], PetscScalar InvF_[], PetscScalar P_[], PetscScalar Pr[], PetscScalar dPr_[], void *ctx)
{
  AppCtx *user = (AppCtx *)ctx;
  PetscErrorCode ierr;
  
  PetscFunctionBegin;

  PetscScalar (*dphi0)[dim] = (PetscScalar (*)[dim]) (dphi0_);
  PetscScalar (*dphi)[dim] = (PetscScalar (*)[dim]) (dphi_);
  PetscScalar (*FHess)[dim][dim] = (PetscScalar (*)[dim][dim]) (FHess_);
  PetscScalar (*F)[dim] = (PetscScalar (*)[dim]) (F_);
  PetscScalar (*InvF)[dim] = (PetscScalar (*)[dim]) (InvF_);
  PetscScalar (*P)[dim] = (PetscScalar (*)[dim]) (P_);
  PetscScalar (*dPr)[dim] = (PetscScalar (*)[dim]) (dPr_);
   
  PetscInt i,j,k,l;
  PetscScalar beta = user->factor;
  PetscScalar e = user->e;
  PetscScalar grad_VolSweSum;
  PetscScalar Trace_PFT;
  PetscScalar Jphi_grad_1[dim];
  PetscScalar Jphi_grad_13[dim];
  PetscScalar Jphi_grad_13p[dim];
  PetscScalar J_grad[dim];
  PetscScalar det_F_grad[dim];
  PetscScalar InvHess[dim][dim][dim];

  // Eq. 35
  
  for(k=0;k<user->nf;k++){
    Trace_PFT = 0.0;
    for(i=0;i<dim;i++){
      for(j=0;j<dim;j++){
	Trace_PFT += P[i][j]*F[i][j];		
      }
    }
    Pr[k] = -((PetscScalar)1/3)*user->omega_phi[k]*pow(user->Jphi,((PetscScalar)-1))*Trace_PFT;
  }
   
  // Eq. 41
  for(i=0;i<dim;i++){
    grad_VolSweSum = 0.0;
    for(j=0;j<user->nf;j++){grad_VolSweSum += user->omega_phi[j]*(dphi[j][i]-dphi0[j][i]);}
    Jphi_grad_1[i] = -pow(user->Jphi,((PetscScalar)-2))*grad_VolSweSum;
  }
  
  // Eq. 42
  for(i=0;i<dim;i++){
    grad_VolSweSum = 0.0;
    for(j=0;j<user->nf;j++){grad_VolSweSum += user->omega_phi[j]*(dphi[j][i]-dphi0[j][i]);}
    Jphi_grad_13[i] = -((PetscScalar)1/3)*pow(user->Jphi,((PetscScalar)-4/3))*grad_VolSweSum;
  }
      
  // Eq. 43
  
  for(i=0;i<dim;i++){
    grad_VolSweSum = 0.0;
    for(j=0;j<user->nf;j++){grad_VolSweSum += user->omega_phi[j]*(dphi[j][i]-dphi0[j][i]);}
    Jphi_grad_13p[i] = ((PetscScalar)1/3)*pow(user->Jphi,((PetscScalar)-2/3))*grad_VolSweSum;
  }
 
  // Eq. 44

  ierr = MaterialJacobianGradient(dim, &F[0][0], &FHess[0][0][0], &J_grad[0]);CHKERRQ(ierr); // I have to complete this function for 3D
  for(i=0;i<dim;i++){
    grad_VolSweSum = 0.0;
    for(j=0;j<user->nf;j++){grad_VolSweSum += user->omega_phi[j]*(dphi[j][i]-dphi0[j][i]);}
    det_F_grad[i] = 0.0;
    det_F_grad[i] += -((PetscScalar)dim/3)*pow(user->Jphi,((PetscScalar)-dim/3 -1))*grad_VolSweSum*det;
    det_F_grad[i] += pow(user->Jphi,((PetscScalar)-dim/3))*J_grad[i];
  } 

  // Eq. 36 
  ierr = HessianInverse(dim, &FHess[0][0][0], &InvF[0][0], &InvHess[0][0][0], NULL);CHKERRQ(ierr);
  PetscScalar temp1,temp2,temp3;
  PetscScalar aux_Term1 = pow(user->Jphi,((PetscScalar)-1/3));
  PetscScalar aux_Term2 = beta*pow((pow(user->Jphi,((PetscScalar)-dim/3)))*(det),((PetscScalar)-beta-1));
  PetscScalar aux_Term3	= pow(user->Jphi,((PetscScalar)1/3));
  PetscScalar aux_Term4 = pow((pow(user->Jphi,((PetscScalar)-dim/3)))*(det),((PetscScalar)-beta));

  for(k=0;k<user->nf;k++){ // component
    for(l=0;l<dim;l++){ // gradient 
      Trace_PFT = 0.0;temp1=0.0;temp2=0.0;temp3=0.0;
      for(i=0;i<dim;i++){
	for(j=0;j<dim;j++){
	  Trace_PFT += P[i][j]*F[i][j];
	  temp1 += Jphi_grad_13[l]*(P[i][j]/aux_Term1)*F[i][j];
	  temp2 += P[i][j]*FHess[i][j][l]; 
	  temp3 += aux_Term1*(Jphi_grad_13[l]*F[i][j] + aux_Term1*FHess[i][j][l])*F[i][j];
	  temp3 += aux_Term1*(aux_Term2*det_F_grad[l]*aux_Term3*InvF[j][i])*F[i][j];
	  temp3 += aux_Term1*(-aux_Term4*Jphi_grad_13p[l]*InvF[j][i])*F[i][j];
	  temp3 += aux_Term1*(-aux_Term4*aux_Term3*InvHess[j][i][l])*F[i][j];	  
	}
      }
      dPr[k][l] = 0.0;
      dPr[k][l] += -((PetscScalar)1/3)*user->omega_phi[k]*e*Jphi_grad_1[l]*Trace_PFT;
      dPr[k][l] += -((PetscScalar)1/3)*user->omega_phi[k]*e*pow(user->Jphi,((PetscScalar)-1))*(temp1+temp2+temp3); 
    }
  }

  //----------------------------------------------------------------------------------------------------------------------------------------------//
    
  PetscFunctionReturn(0);
}
  
PETSC_STATIC_INLINE
PetscReal DOT(PetscInt dim,const PetscReal a[],const PetscReal b[])
{
  PetscInt i; PetscReal s = 0.0;
  for (i=0; i<dim; i++) s += a[i]*b[i];
  return s;
}

#undef  __FUNCT__
#define __FUNCT__ "ReactionsRates"
PetscErrorCode ReactionsRates(PetscReal phi[], PetscReal reaction_rate[], void *ctx)
{
  AppCtx *user = (AppCtx *)ctx;

  PetscFunctionBegin;

  PetscInt i;  
  PetscReal forward, backward;
  PetscInt nr = user->nr;
  PetscInt np = user->np;

  forward = 1.0;  
  for(i=0; i<nr; i++){ forward *= pow(phi[i],user->alpha[i]); }
  reaction_rate[0] = (user->kplus)*forward;

  backward = 1.0;
  for(i=0; i<np; i++){ backward *= pow(phi[i+nr],user->beta[i]); }
  reaction_rate[1] = (user->kminus)*backward;
  
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "dimensionless_ChemicalPotential_bulk"
PetscErrorCode dimensionless_ChemicalPotential_bulk(PetscInt dim, PetscScalar phi[], PetscScalar dphi_[], PetscScalar eta[], PetscScalar deta_[], void *ctx) 
{ 
  AppCtx *user = (AppCtx *)ctx;

  PetscFunctionBegin;

  PetscInt nf = user->nf;
  PetscInt a,b,i;
  PetscScalar sumfactor, sumFactor[dim];
  const PetscReal (*theta_c)[nf] = (typeof(theta_c)) user->theta_c;
  PetscScalar (*dphi)[dim]     = (PetscScalar (*)[dim])  (dphi_);
  PetscScalar (*deta)[dim]     = (PetscScalar (*)[dim])  (deta_);

  if (eta) {            
    for(a=0; a<nf; a++){
      sumfactor = 0.0;
      for(b=0; b<nf; b++){
        sumfactor += theta_c[a][b]*phi[b];
      }
      eta[a] = 0.5*(log(phi[a]) + 1.0) + 2.0*sumfactor;
    }
  }

  if (deta_) {
    for(a=0; a<nf; a++){
      for(i=0; i<dim; i++){ 
        sumFactor[i] = 0.0; 
        for(b=0; b<nf; b++){
          sumFactor[i] += theta_c[a][b]*dphi[b][i];
        }
        deta[a][i] = 0.5*dphi[a][i]/phi[a] + 2.0*sumFactor[i];
      }
    }
  }

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "dimensionless_ChemicalPotential_interface"
PetscErrorCode dimensionless_ChemicalPotential_interface(PetscScalar delphi[], PetscScalar eta[], void *ctx) 
{

  AppCtx *user = (AppCtx *)ctx;

  PetscFunctionBegin;

  PetscInt nf = user->nf;
  PetscInt a,b;
  PetscScalar sumfactor;
  const PetscReal (*cahn)[nf] = (typeof(cahn)) user->cahn;
  const PetscReal (*lsigma)[nf] = (typeof(lsigma)) user->lsigma;
  
  if (eta) {
    for(a=0; a<nf; a++){ 
      sumfactor = 0.0;
      for(b=0; b<nf; b++){
        sumfactor += cahn[a][b]*lsigma[a][b]*delphi[b];
      }
      eta[a] = -sumfactor;
    }
  }

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "dimensionless_diffusion"
PetscErrorCode dimensionless_diffusion(PetscInt dim, PetscScalar phi[], PetscScalar dphi_[], PetscScalar L_[], PetscScalar dL_[], void *ctx)
{
  AppCtx *user = (AppCtx *)ctx;

  PetscFunctionBegin;

  PetscInt nf = user->nf;
  PetscInt a,b,i;

  PetscScalar (*dphi)[dim]    = (PetscScalar (*)[dim])      (dphi_);
  PetscScalar (*L)[nf]        = (PetscScalar (*)[nf])       (L_);
  PetscScalar (*dL)[nf][dim]  = (PetscScalar (*)[nf][dim])  (dL_);

  for(a=0; a<nf; a++){
    for(b=0; b<nf; b++){
      L[a][b] = -user->Diff[a]*phi[a]*phi[b];
      if(a == b){
        L[a][b] += user->Diff[a]*phi[a];
      }
      for(i=0; i<dim; i++){
        dL[a][b][i] = - user->Diff[a]*(dphi[a][i]*phi[b] + phi[a]*dphi[b][i]);
        if(a == b){
          dL[a][b][i] += user->Diff[a]*dphi[a][i];
        }
      }
    }
  }

  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "FormInitialCondition"
PetscErrorCode FormInitialCondition(IGA iga, Vec U, AppCtx *user)
{
  MPI_Comm       comm;
  PetscRandom    rctx;
  PetscErrorCode ierr;
  PetscFunctionBegin;
  ierr = IGAGetComm(iga,&comm);CHKERRQ(ierr);
  ierr = PetscRandomCreate(comm,&rctx);CHKERRQ(ierr);
  ierr = PetscRandomSetFromOptions(rctx);CHKERRQ(ierr);
  ierr = PetscRandomSetInterval(rctx,user->phibar-0.05,user->phibar+0.05);CHKERRQ(ierr);
  ierr = PetscRandomSeed(rctx);CHKERRQ(ierr);
  ierr = VecSetRandom(U,rctx);CHKERRQ(ierr);
  ierr = PetscRandomDestroy(&rctx);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "IniCondSystem"
PetscErrorCode IniCondSystem(IGAPoint p,PetscScalar *K,PetscScalar *F,void *ctx)
{
  PetscInt a,b,c;
  PetscInt nen    = p->nen;
  PetscInt dof    = p->dof;
  PetscReal pos[p->dim];
  IGAPointFormPoint(p,pos);
  
  PetscReal x  = pos[0];
  PetscReal y  = pos[1];
  
  PetscScalar h = 0.02;
  PetscScalar a1 = tanh((pow(x-0.375,2.0) + pow(y-0.65,2.0) - pow(0.15,2.0))*((pow(h,-1.0))/(0.3+h)))*0.499 -0.5;
  PetscScalar a2 = tanh((pow(x-0.375,2.0) + pow(y-0.30,2.0) - pow(0.10,2.0))*((pow(h,-1.0))/(0.2+h)))*0.499 -0.5;
  PetscScalar phi0 = a1+a2+1.0;

  /*
  PetscReal phi0;
  PetscReal h1 = ((PetscScalar)1/8); PetscReal k1 = ((PetscScalar)2/8); PetscReal r1 = ((PetscScalar)1/32); PetscReal c1 = pow(x-h1,2.0) + pow(y-k1,2.0);
  PetscReal h2 = ((PetscScalar)1/8); PetscReal k2 = ((PetscScalar)1/8); PetscReal r2 = ((PetscScalar)1/32); PetscReal c2 = pow(x-h2,2.0) + pow(y-k2,2.0);
  PetscReal h3 = ((PetscScalar)2/8); PetscReal k3 = ((PetscScalar)2/8); PetscReal r3 = ((PetscScalar)1/32); PetscReal c3 = pow(x-h3,2.0) + pow(y-k3,2.0);
  PetscReal h4 = ((PetscScalar)2/8); PetscReal k4 = ((PetscScalar)4/8); PetscReal r4 = ((PetscScalar)1/16); PetscReal c4 = pow(x-h4,2.0) + pow(y-k4,2.0);
  PetscReal h5 = ((PetscScalar)3/8); PetscReal k5 = ((PetscScalar)5/8); PetscReal r5 = ((PetscScalar)1/32); PetscReal c5 = pow(x-h5,2.0) + pow(y-k5,2.0); 
  PetscReal h6 = ((PetscScalar)3/8); PetscReal k6 = ((PetscScalar)3/8); PetscReal r6 = ((PetscScalar)1/16); PetscReal c6 = pow(x-h6,2.0) + pow(y-k6,2.0);
  PetscReal h7 = ((PetscScalar)4/8); PetscReal k7 = ((PetscScalar)3/8); PetscReal r7 = ((PetscScalar)1/32); PetscReal c7 = pow(x-h7,2.0) + pow(y-k7,2.0);
  PetscReal h8 = ((PetscScalar)4/8); PetscReal k8 = ((PetscScalar)1/8); PetscReal r8 = ((PetscScalar)1/16); PetscReal c8 = pow(x-h8,2.0) + pow(y-k8,2.0);
  */
//  PetscReal h9 = ((PetscScalar)5/8); PetscReal k9 = ((PetscScalar)2/8); PetscReal r9 = ((PetscScalar)1/32); PetscReal c9 = pow(x-h9,2.0) + pow(y-k9,2.0);
  // PetscReal h10 = ((PetscScalar)5/8); PetscReal k10 = ((PetscScalar)4/8); PetscReal r10 = ((PetscScalar)1/32); PetscReal c10 = pow(x-h10,2.0) + pow(y-k10,2.0);
  //  PetscReal h11 = ((PetscScalar)5/8); PetscReal k11 = ((PetscScalar)7/8); PetscReal r11 = ((PetscScalar)1/16); PetscReal c11 = pow(x-h11,2.0) + pow(y-k11,2.0);
  // PetscReal h12 = ((PetscScalar)6/8); PetscReal k12 = ((PetscScalar)5/8); PetscReal r12 = ((PetscScalar)1/32); PetscReal c12 = pow(x-h12,2.0) + pow(y-k12,2.0);
  // PetscReal h13 = ((PetscScalar)6/8); PetscReal k13 = ((PetscScalar)3/8); PetscReal r13 = ((PetscScalar)1/32); PetscReal c13 = pow(x-h13,2.0) + pow(y-k13,2.0);
  //  PetscReal h14 = ((PetscScalar)7/8); PetscReal k14 = ((PetscScalar)7/8); PetscReal r14 = ((PetscScalar)1/32); PetscReal c14 = pow(x-h14,2.0) + pow(y-k14,2.0);
  // PetscReal h15 = ((PetscScalar)7/8); PetscReal k15 = ((PetscScalar)5/8); PetscReal r15 = ((PetscScalar)1/32); PetscReal c15 = pow(x-h15,2.0) + pow(y-k15,2.0);
  //  PetscReal h16 = ((PetscScalar)7/8); PetscReal k16 = ((PetscScalar)4/8); PetscReal r16 = ((PetscScalar)1/32); PetscReal c16 = pow(x-h16,2.0) + pow(y-k16,2.0);
  // PetscReal h17 = ((PetscScalar)7/8); PetscReal k17 = ((PetscScalar)2/8); PetscReal r17 = ((PetscScalar)1/32); PetscReal c17 = pow(x-h17,2.0) + pow(y-k17,2.0);
 
  /*
  if(c1<r1*r1||c2<r2*r2||c3<r3*r3||c4<r4*r4||c5<r5*r5||c6<r6*r6||c7<r7*r7||c8<r8*r8){
    phi0 = 0.8;
  }
  else{
    phi0 = 0.2;
  }
  */
  
  const PetscReal *N0 = (typeof(N0)) p->shape[0];
  PetscScalar (*Fa)[dof] = (PetscScalar (*)[dof])F;
  PetscScalar (*Ka)[dof][nen][dof] = (PetscScalar (*)[dof][nen][dof])K;
  for(a=0; a<nen; a++) {
    Fa[a][0] = N0[a] * phi0;
    Fa[a][1] = N0[a] * 0.0;
    Fa[a][2] = N0[a] * 0.0;
    for(b=0; b<nen; b++) {
      for (c=0; c<dof; c++){
        Ka[a][c][b][c] = N0[a]*N0[b];
      }
    }
  }
  return 0;
}

#undef  __FUNCT__
#define __FUNCT__ "IniCondSystem3bubbles2D"
PetscErrorCode IniCondSystem3bubbles(IGAPoint p,PetscScalar *K,PetscScalar *F,void *ctx)
{
  PetscInt a,b,c;
  PetscInt nen    = p->nen;
  PetscInt dof    = p->dof;
  PetscReal pos[p->dim];
  IGAPointFormPoint(p,pos);
  
  PetscReal x  = pos[0];
  PetscReal y  = pos[1];
  
  PetscScalar h = 0.02;
  PetscScalar a1 = tanh((pow(x-0.275,2.0) + pow(y-0.65,2.0) - pow(0.15,2.0))*((pow(h,-1.0))/(0.3+h)))*0.2;
  PetscScalar a2 = tanh((pow(x-0.375,2.0) + pow(y-0.30,2.0) - pow(0.10,2.0))*((pow(h,-1.0))/(0.2+h)))*0.2;
  PetscScalar a3 = tanh((pow(x-0.575,2.0) + pow(y-0.50,2.0) - pow(0.07,2.0))*((pow(h,-1.0))/(0.2+h)))*0.2;

  PetscScalar phi0 = 0.5 - a1 - a2;
  PetscScalar phi1 = 0.3 - a3;
  const PetscReal *N0 = (typeof(N0)) p->shape[0];
  PetscScalar (*Fa)[dof]           = (PetscScalar (*)[dof])F;
  PetscScalar (*Ka)[dof][nen][dof] = (PetscScalar (*)[dof][nen][dof])K;
  for(a=0; a<nen; a++) {
    Fa[a][0] = N0[a] * phi0;
    Fa[a][1] = N0[a] * phi1;
    Fa[a][2] = N0[a] * 0.0;
    Fa[a][3] = N0[a] * 0.0;
    for(b=0; b<nen; b++) {
      for (c=0; c<dof; c++){
        Ka[a][c][b][c] = N0[a]*N0[b];
      }
    }
  }
  return 0;
}

#undef  __FUNCT__
#define __FUNCT__ "IniCondSystem3bubbles3D"
PetscErrorCode IniCondSystem3bubbles3D(IGAPoint p,PetscScalar *K,PetscScalar *F,void *ctx)
{
  PetscInt a,b,c;
  PetscInt nen    = p->nen;
  PetscInt dof    = p->dof;
  PetscReal pos[p->dim];
  IGAPointFormPoint(p,pos);

  PetscReal x  = pos[0];
  PetscReal y  = pos[1];
  PetscReal z  = pos[2];
  PetscReal r1 = 0.2;
  PetscReal r2 = 0.1;
  PetscReal r3 = 0.075;

  PetscReal phi0, phi1;
  PetscReal spherein1 = (x-0.25)*(x-0.25) + (y-0.25)*(y-0.25) + (z-0.25)*(z-0.25);
  PetscReal spherein2 = (x-0.75)*(x-0.75) + (y-0.75)*(y-0.75) + (z-0.75)*(z-0.75);
  PetscReal spherein3 = (x-0.75)*(x-0.75) + (y-0.3)*(y-0.3) + (z-0.3)*(z-0.3);

  if(spherein1 < r1*r1 || spherein3 < r3*r3){
    phi0 = 0.8;
    phi1 = 0.2;
  }
  else if(spherein2 < r2*r2){
    phi1 = 0.8;
    phi0 = 0.2;
  }
  else{
    phi0 = 0.2;
    phi1 = 0.2;
  }

  const PetscReal *N0 = (typeof(N0)) p->shape[0];
  PetscScalar (*Fa)[dof]           = (PetscScalar (*)[dof])F;
  PetscScalar (*Ka)[dof][nen][dof] = (PetscScalar (*)[dof][nen][dof])K;
  for(a=0; a<nen; a++) {
    Fa[a][0] = N0[a] * phi0;
    Fa[a][1] = N0[a] * phi1;
    for(b=0; b<nen; b++) {
      for (c=0; c<dof; c++){
	Ka[a][c][b][c] = N0[a]*N0[b];
      }
    }
  }
  return 0;
}

PetscErrorCode ResidualPrimal (IGAPoint p,PetscReal dt,
			       PetscReal shift, const PetscScalar *V,
			       PetscReal t,const PetscScalar *U,
			       PetscScalar *Re,void *ctx);
#undef  __FUNCT__
#define __FUNCT__ "ResidualPrimal"
PetscErrorCode ResidualPrimal(IGAPoint p,PetscReal dt,
			      PetscReal shift, const PetscScalar *V,
			      PetscReal t,const PetscScalar *U,
			      PetscScalar *Re,void *ctx)
{
  AppCtx *user = (AppCtx *)ctx;
  PetscErrorCode ierr;

  PetscFunctionBegin;
  
  PetscInt a,c,d,i,j,k;
  PetscInt nen = p->nen;
  PetscInt dim = p->dim;
  PetscInt nf  = user->nf;

  PetscScalar phi_t[nf];
  IGAPointFormValue(p,V,&phi_t[0]);

  PetscInt nReactions = user->nReactions;
  const PetscReal (*stch_matrix)[nReactions] = (typeof(stch_matrix)) user->stoichiometry_matrix;
    
  //Assembling vectors for phases;
  //Gathering values from U0 and U
  
  PetscScalar u[nf-1+dim];
  PetscScalar du[nf-1+dim][dim];
  IGAPointFormValue(p,U,&u[0]);
  IGAPointFormGrad(p,U,&du[0][0]);

  IGAProbe Unit = user->Unit;
  PetscScalar u0[nf-1+dim];
  PetscScalar du0[nf-1+dim][dim];
  
  IGAProbeSetPoint(Unit,p->point);
  IGAProbeFormValue(Unit,&u0[0]);
  IGAProbeFormGrad(Unit,&du0[0][0]);
  
  PetscScalar phi0[nf], dphi0[nf][dim];
  PetscScalar phi[nf], dphi[nf][dim];
  
  for(c=0;c<nf-1;c++){ // component
    phi0[c] = u0[c];
    phi[c] = u[c];
    for(i=0;i<dim;i++){  //gradient
      dphi0[c][i] = du0[c][i];
      dphi[c][i] = du[c][i];
    }
  }
  
  PetscScalar d2u[nf-1+dim][dim][dim];
  PetscScalar d2phi[nf][dim][dim];
  IGAPointFormHess (p,U,&d2u[0][0][0]);

  // Splitting the Hessian for phases
  for(c=0;c<nf-1;c++){
    for(i=0;i<dim;i++){
      for(j=0;j<dim;j++){
	d2phi[c][i][j] = d2u[c][i][j];
      }
    }
  }

  PetscScalar delphi[nf];
  //Gather the laplacians
  for(c=0; c<nf-1; c++){   
    delphi[c] = 0.0;
    for(i=0; i<dim; i++){
      delphi[c] += d2phi[c][i][i];
    }
  }
  
  //Applying Lattice constrain $sum_{a=1}^{nf} = 1.0$
  phi0[nf-1] = 1.0;
  phi[nf-1] = 1.0;
  delphi[nf-1] = 0.0;
  for(i=0; i<dim; i++){ dphi0[nf-1][i] = 0.0; dphi[nf-1][i] = 0.0;}

  //Computing the component nf
  for(c=0; c<nf-1; c++){ 
      phi0[nf-1] += -phi0[c];
      phi[nf-1] += -phi[c];
      delphi[nf-1] += -delphi[c];
      for(i=0; i<dim; i++){ dphi0[nf-1][i] += -dphi0[c][i]; dphi[nf-1][i] += -dphi[c][i];}
  }
  
  PetscScalar dd[dim][dim];
  for(i=0;i<dim;i++){
    for(j=0;j<dim;j++){
      dd[i][j] = du[nf-1+i][j];
    }
  }
 
  PetscScalar d2d[dim][dim][dim];
  for(k=0;k<dim;k++){
    for(i=0;i<dim;i++){
      for(j=0;j<dim;j++){
	d2d[k][i][j] = d2u[nf-1+k][i][j];
      }
    }
  }

  PetscScalar F[dim][dim];
  PetscScalar InvF[dim][dim];
  PetscScalar J;
  PetscScalar P[dim][dim];
  PetscScalar Pr[nf];
  PetscScalar dPr[nf][dim];
  
  ierr = TensorOperations(dim, &J, &dd[0][0], &F[0][0], &InvF[0][0],user);CHKERRQ(ierr);
  ierr = FirstPiolaKirchkoff(dim, phi0, phi, J, &F[0][0], &InvF[0][0], &P[0][0], user);CHKERRQ(ierr);
  ierr = MaterialPressureGradient(dim, J, phi0, phi, &dphi0[0][0], &dphi[0][0], &d2d[0][0][0],
				  &F[0][0], &InvF[0][0], &P[0][0], Pr, &dPr[0][0], user);CHKERRQ(ierr);
  
  PetscScalar M[nf][nf], dM[nf][nf][dim];
  ierr = dimensionless_diffusion(dim, &phi[0], &dphi[0][0], &M[0][0], &dM[0][0][0], user);CHKERRQ(ierr);
  
  PetscReal mu_iface[nf];
  PetscReal di_mu_bulk[nf][dim];
  PetscScalar InvHessR[dim];
  PetscReal reaction_rate[nReactions];
  PetscReal reaction_term[nf];
  
  ierr = dimensionless_ChemicalPotential_bulk(dim, phi, &dphi[0][0], NULL, &di_mu_bulk[0][0], user);CHKERRQ(ierr);
  ierr = dimensionless_ChemicalPotential_interface(delphi, &mu_iface[0], user);CHKERRQ(ierr);

  ierr = HessianInverse(dim, &d2d[0][0][0], &InvF[0][0], NULL, &InvHessR[0]);CHKERRQ(ierr);
  ierr = ReactionsRates(phi, &reaction_rate[0], user);CHKERRQ(ierr);
  
  PetscReal sum_reaction; 
  for (i=0; i<nf; i++){
    sum_reaction = 0.0;
    for (j=0; j<nReactions; j++){sum_reaction += stch_matrix[i][j]*reaction_rate[j]; }
    reaction_term[i] = sum_reaction;
  }
    
  const PetscReal  *N0            = (typeof(N0)) p->shape[0];
  const PetscReal (*N1)[dim]      = (typeof(N1)) p->shape[1];
  const PetscReal (*N2)[dim][dim] = (typeof(N2)) p->shape[2];

  PetscScalar (*R)[nf-1+dim] = (PetscScalar (*)[nf-1+dim])Re;

  PetscReal   delN;
  PetscScalar   M_times_mu_iface;
  PetscScalar   M_times_di_mu_bulk[dim];
  PetscScalar   P_times_di_mu_bulk[dim];
  PetscScalar   di_M_times_mu_iface[dim];
  PetscScalar TensorVector;
  PetscScalar Base_Inv[dim];
  
  for (a=0; a<nen; a++) {
    
    delN = 0.0; 
    for(i=0; i<dim; i++){
      for(j=0; j<dim; j++){
	for(k=0; k<dim; k++){
	  delN += N2[a][i][j]*InvF[j][k]*InvF[i][k];
	}
      }
    }
    
    for(i=0;i<dim;i++){
      Base_Inv[i]=0.0;
      for(j=0;j<dim;j++){
	Base_Inv[i]+=N1[a][j]*InvF[j][i];
      }
    }
    
    for(c=0; c < nf-1; c++){
      
      M_times_mu_iface = 0.0;
      for(d=0; d < nf-1; d++){ M_times_mu_iface += M[c][d]*(mu_iface[d] - mu_iface[nf-1]);}
      
      for(i=0;i<dim;i++){
	M_times_di_mu_bulk[i]  = 0.0;
	di_M_times_mu_iface[i] = 0.0;
	P_times_di_mu_bulk[i] = 0.0;
	for(d=0;d<nf-1;d++){
	  for(j=0;j<dim;j++){
	    M_times_di_mu_bulk[i]  += M[c][d]*(di_mu_bulk[d][j] - di_mu_bulk[nf-1][j])*InvF[j][i];
	    di_M_times_mu_iface[i] += dM[c][d][j]*InvF[j][i]*(mu_iface[d] - mu_iface[nf-1]);
	    P_times_di_mu_bulk[i]  += M[c][d]*(dPr[d][j] - dPr[nf-1][j])*InvF[j][i];
	  }
	}
      }
      
      R[a][c] = N0[a]*phi_t[c] - N0[a]*reaction_term[c] + DOT(dim,Base_Inv,P_times_di_mu_bulk)*J + DOT(dim,Base_Inv,M_times_di_mu_bulk)*J - delN*M_times_mu_iface*J - DOT(dim,N1[a],di_M_times_mu_iface)*J - DOT(dim,N1[a],InvHessR)*M_times_mu_iface*J;
    }
  
    
    for(i=0;i<dim;i++){
      TensorVector = 0.0; for(j=0;j<dim;j++){TensorVector+=N1[a][j]*P[i][j];}
      R[a][nf-1+i] = TensorVector;
    }
  }
    
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "OutputMonitor"
PetscErrorCode OutputMonitor(TS ts,PetscInt step,PetscReal t,Vec U,void *mctx)
{
    char         filename[PETSC_MAX_PATH_LEN];
    PetscInt     saveRate = 1;
    PetscErrorCode ierr;
    PetscFunctionBegin;
    //OMonitor *monit = (OMonitor*)mctx;
    AppCtx *user = (AppCtx *)mctx;
    if((step + user->init) % saveRate == 0){
      IGA iga;
      ierr = PetscObjectQuery((PetscObject)ts,"IGA",(PetscObject*)&iga);CHKERRQ(ierr);
      ierr = PetscSNPrintf(filename,sizeof(filename),"ch-%d.dat",(int)step + user->init);CHKERRQ(ierr);
      ierr = IGAWriteVec(iga,U,filename);CHKERRQ(ierr);
    }
    PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Init2PhasesParameters"
PetscErrorCode Init2PhasesParameters(AppCtx *user)
{

  PetscErrorCode  ierr;
  PetscInt i,j;
  PetscPrintf(PETSC_COMM_WORLD,"Setting up the parameters for 3 phases example. \n\n");
  
  user->nr = 1;
  user->np = 1;
  user->nf = user->nr + user->np;
  PetscPrintf(PETSC_COMM_WORLD,"#Reactants: %d, #Product: %d, #Number of phases: %d \n\n", user->nr,user->np,user->nf);

  user->factor = 0.5;
  user->Psi0 = 50000.0;
  
  PetscPrintf(PETSC_COMM_WORLD,"Symmetry factor= %8.4g \n",user->factor);
  //Allocating and initializing omega
  ierr = PetscMalloc1(user->nf,&(user->omega_phi));CHKERRQ(ierr);
  user->omega_phi[0] = 0.0828789;
  user->omega_phi[1] = 0.0235468;

  PetscPrintf(PETSC_COMM_WORLD,"swelling parameters:  \n");
  for(i=0; i<user->nf; i++){
    PetscPrintf(PETSC_COMM_WORLD,"Omega_phi[%d]= %8.7g\n",i,user->omega_phi[i]);
  }

  //Allocating and initializing diffusion
  ierr = PetscMalloc1(user->nf,&(user->Diff));CHKERRQ(ierr);
  user->Diff[0] = 10000.0;
  user->Diff[1] = 10000.0;
  PetscPrintf(PETSC_COMM_WORLD,"dimensionless diffusion coeffcient:  \n");
  for(i=0; i<user->nf; i++){
    PetscPrintf(PETSC_COMM_WORLD,"Diff[%d]= %8.7g\n",i,user->Diff[i]);
  }

  //Allocating and initializing alpha
  ierr = PetscMalloc1(user->nr,&(user->alpha));CHKERRQ(ierr);
  user->alpha[0] = 1.0;
  PetscPrintf(PETSC_COMM_WORLD,"Stochiometric coefficients of reactants:  \n");
  for(i=0; i<user->nr; i++){
    PetscPrintf(PETSC_COMM_WORLD,"alpha[%d]= %8.4g ",i,user->alpha[i]);
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n\n");  

  //Allocating and initializing beta
  ierr = PetscMalloc1(user->np,&(user->beta));CHKERRQ(ierr);
  user->beta[0] = 1.0;
  PetscPrintf(PETSC_COMM_WORLD,"Stochiometric coefficients of products:  \n");
  for(i=0; i<user->np; i++){
    PetscPrintf(PETSC_COMM_WORLD,"beta[%d]= %8.4g ",i,user->beta[i]);
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n\n");  

  //Allocating and initializing cahn
  ierr = PetscMalloc1((user->nf)*(user->nf),&(user->cahn));CHKERRQ(ierr);
  PetscReal (*Cahn)[user->nf] = (PetscReal (*)[user->nf])  (user->cahn);

  Cahn[0][0] = 0.01;     Cahn[0][1] = 0.0;
  Cahn[1][0] = 0.0;     Cahn[1][1] = 0.01;

  PetscPrintf(PETSC_COMM_WORLD,"Cahn number:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<user->nf; j++){
      PetscPrintf(PETSC_COMM_WORLD,"cahn[%d][%d]= %8.4g ",i,j,Cahn[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");

  //Allocating and initializing gamma
  ierr = PetscMalloc1((user->nf)*(user->nf),&(user->lsigma));CHKERRQ(ierr);
  PetscReal (*Lsigma)[user->nf] = (PetscReal (*)[user->nf])  (user->lsigma);

  Lsigma[0][0] = 0.0817;     Lsigma[0][1] = 0.0;
  Lsigma[1][0] = 0.0;     Lsigma[1][1] = 0.0817;

  PetscPrintf(PETSC_COMM_WORLD,"dimensionless interfacial tension:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<user->nf; j++){
      PetscPrintf(PETSC_COMM_WORLD,"lsigma[%d][%d]= %8.4g ",i,j,Lsigma[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");

  //Allocating and initializing critical theta
  ierr = PetscMalloc1((user->nf)*(user->nf),&(user->theta_c));CHKERRQ(ierr);
  PetscReal (*Theta_c)[user->nf] = (PetscReal (*)[user->nf])  (user->theta_c);

  Theta_c[0][0] = 0.0;     Theta_c[0][1] = 1.27;
  Theta_c[1][0] = 1.27;     Theta_c[1][1] = 0.0;

  PetscPrintf(PETSC_COMM_WORLD,"critical theta:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<user->nf; j++){
      PetscPrintf(PETSC_COMM_WORLD,"theta_c[%d][%d]= %8.4g ",i,j,Theta_c[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");

  user->kappa  = 3000.0; PetscPrintf(PETSC_COMM_WORLD,"kappa: %8.4g \n",user->kappa);
  user->delta  = 9000.0; PetscPrintf(PETSC_COMM_WORLD,"delta: %8.4g \n",user->delta);
  user->zeta   = 1.0; PetscPrintf(PETSC_COMM_WORLD,"zeta:  %8.4g \n",user->zeta);
  user->kplus  = 0.0; PetscPrintf(PETSC_COMM_WORLD,"kplus:  %8.4g \n",user->kplus);
  user->kminus = 0.0; PetscPrintf(PETSC_COMM_WORLD,"kminus:  %8.4g \n",user->kminus);
  user->e  = 4400.0; PetscPrintf(PETSC_COMM_WORLD,"e:  %8.4g \n",user->e);
  user->l = 1.0; PetscPrintf(PETSC_COMM_WORLD,"l:  %8.4g \n",user->l);
  
  user->phibar    = 1./(user->nf);      /* average concentration            */
  PetscPrintf(PETSC_COMM_WORLD,"phibar: %4.3g \n",user->phibar);
  
  user->eps       = 0.03;               /* thickness interface parameter    */
  user->C         = 1000.0;
  user->M0        = 0.5;
  user->L0        = 1.0;                /* Length Scale                     */

  // chemical reaction parameters

  user->nReactions = 2;

  ierr = PetscMalloc1((user->nf)*(2),&(user->stoichiometry_matrix));CHKERRQ(ierr);
  PetscReal (*Stoichiometry_matrix)[2] = (PetscReal (*)[2])  (user->stoichiometry_matrix);

  Stoichiometry_matrix[0][0] = -1.0; Stoichiometry_matrix[0][1] = 1.0;
  Stoichiometry_matrix[1][0] = 1.0; Stoichiometry_matrix[1][1] = -1.0;
  
  PetscPrintf(PETSC_COMM_WORLD,"Stoichiometry matrix:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<2; j++){
      PetscPrintf(PETSC_COMM_WORLD,"stoichiometry_matrix[%d][%d]= %1.0f ",i,j,Stoichiometry_matrix[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "Init3PhasesParameters"
PetscErrorCode Init3PhasesParameters(AppCtx *user)
{

  PetscErrorCode  ierr;
  PetscInt i,j;
  PetscPrintf(PETSC_COMM_WORLD,"Setting up the parameters for 3 phases example. \n\n");
  
  user->nr = 2;
  user->np = 1;
  user->nf = user->nr + user->np;
  PetscPrintf(PETSC_COMM_WORLD,"#Reactants: %d, #Product: %d, #Number of phases: %d \n\n", user->nr,user->np,user->nf);

  user->factor = 0.5;
  user->Psi0 = 50000.0;
  PetscPrintf(PETSC_COMM_WORLD,"Symmetry factor= %8.4g \n",user->factor);

  //Allocating and initializing omega
  ierr = PetscMalloc1(user->nf,&(user->omega_phi));CHKERRQ(ierr);
  user->omega_phi[0] = 0.03828789;
  user->omega_phi[1] = 0.02235468;
  user->omega_phi[2] = 0.01654562;
  PetscPrintf(PETSC_COMM_WORLD,"swelling parameters:  \n");
  for(i=0; i<user->nf; i++){
    PetscPrintf(PETSC_COMM_WORLD,"Omega_phi[%d]= %8.7g\n",i,user->omega_phi[i]);
  }

  //Allocating and initializing diffusion 
  ierr = PetscMalloc1(user->nf,&(user->Diff));CHKERRQ(ierr);
  user->Diff[0] = 10000.0;
  user->Diff[1] = 10000.0;
  user->Diff[2] = 10000.0;
  PetscPrintf(PETSC_COMM_WORLD,"dimensionless diffusion coeffcient:  \n");
  for(i=0; i<user->nf; i++){
    PetscPrintf(PETSC_COMM_WORLD,"Diff[%d]= %8.7g\n",i,user->Diff[i]);
  }
  
  //Allocating and initializing alpha
  ierr = PetscMalloc1(user->nr,&(user->alpha));CHKERRQ(ierr);
  user->alpha[0] = 1.0;
  user->alpha[1] = 1.0;
  PetscPrintf(PETSC_COMM_WORLD,"Stochiometric coefficients of reactants:  \n");
  for(i=0; i<user->nr; i++){
    PetscPrintf(PETSC_COMM_WORLD,"alpha[%d]= %8.4g ",i,user->alpha[i]);
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n\n");  

  //Allocating and initializing beta
  ierr = PetscMalloc1(user->np,&(user->beta));CHKERRQ(ierr);
  user->beta[0] = 2.0;
  PetscPrintf(PETSC_COMM_WORLD,"Stochiometric coefficients of products:  \n");
  for(i=0; i<user->np; i++){
    PetscPrintf(PETSC_COMM_WORLD,"beta[%d]= %8.4g ",i,user->beta[i]);
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n\n");  

  //Allocating and initializing cahn
  ierr = PetscMalloc1((user->nf)*(user->nf),&(user->cahn));CHKERRQ(ierr);
  PetscReal (*Cahn)[user->nf] = (PetscReal (*)[user->nf])  (user->cahn);

  Cahn[0][0] = 0.01;     Cahn[0][1] = 0.0; Cahn[0][2] = 0.0;
  Cahn[1][0] = 0.0;     Cahn[1][1] = 0.01; Cahn[1][2] = 0.0;
  Cahn[2][0] = 0.0;     Cahn[2][1] = 0.0; Cahn[2][2] = 0.01;
  
  PetscPrintf(PETSC_COMM_WORLD,"Cahn number:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<user->nf; j++){
      PetscPrintf(PETSC_COMM_WORLD,"cahn[%d][%d]= %8.4g ",i,j,Cahn[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");

  //Allocating and initializing gamma
  ierr = PetscMalloc1((user->nf)*(user->nf),&(user->lsigma));CHKERRQ(ierr);
  PetscReal (*Lsigma)[user->nf] = (PetscReal (*)[user->nf])  (user->lsigma);

  Lsigma[0][0] = 0.0817;     Lsigma[0][1] = 0.0; Lsigma[0][2] = 0.0;
  Lsigma[1][0] = 0.0;     Lsigma[1][1] = 0.0817; Lsigma[1][2] = 0.0;
  Lsigma[2][0] = 0.0;     Lsigma[2][1] = 0.0; Lsigma[2][2] = 0.0817;
  
  PetscPrintf(PETSC_COMM_WORLD,"dimensionless interfacial tension:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<user->nf; j++){
      PetscPrintf(PETSC_COMM_WORLD,"lsigma[%d][%d]= %8.4g ",i,j,Lsigma[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");

  //Allocating and initializing critical theta
  ierr = PetscMalloc1((user->nf)*(user->nf),&(user->theta_c));CHKERRQ(ierr);
  PetscReal (*Theta_c)[user->nf] = (PetscReal (*)[user->nf])  (user->theta_c);

  Theta_c[0][0] = 0.0;     Theta_c[0][1] = 1.1; Theta_c[0][2] = 1.1;
  Theta_c[1][0] = 1.1;     Theta_c[1][1] = 0.0; Theta_c[1][2] = 1.1;
  Theta_c[2][0] = 1.1;     Theta_c[2][1] = 1.1; Theta_c[2][2] = 0.0;
  
  PetscPrintf(PETSC_COMM_WORLD,"critical theta:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<user->nf; j++){
      PetscPrintf(PETSC_COMM_WORLD,"theta_c[%d][%d]= %8.4g ",i,j,Theta_c[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");
  
  user->kappa  = 3000.0; PetscPrintf(PETSC_COMM_WORLD,"kappa: %8.4g \n",user->kappa);
  user->delta  = 9000.0; PetscPrintf(PETSC_COMM_WORLD,"delta: %8.4g \n",user->delta);
  user->zeta   = 1.0; PetscPrintf(PETSC_COMM_WORLD,"zeta:  %8.4g \n",user->zeta);
  user->kplus  = 10000.0; PetscPrintf(PETSC_COMM_WORLD,"kplus:  %8.4g \n",user->kplus);
  user->kminus = 0.0; PetscPrintf(PETSC_COMM_WORLD,"kminus:  %8.4g \n",user->kminus);
  user->e  = 4400.0; PetscPrintf(PETSC_COMM_WORLD,"e:  %8.4g \n",user->e);
  user->l = 1.0; PetscPrintf(PETSC_COMM_WORLD,"l:  %8.4g \n",user->l);

  user->phibar    = 1./(user->nf);      /* average concentration            */
  PetscPrintf(PETSC_COMM_WORLD,"phibar: %4.3g \n",user->phibar);
  
  user->eps       = 0.03;               /* thickness interface parameter    */
  user->C         = 1000.0;
  user->M0        = 0.5;
  user->L0        = 1.0;                /* Length Scale                     */

  // chemical reaction parameters

  user->nReactions = 2;

  ierr = PetscMalloc1((user->nf)*(2),&(user->stoichiometry_matrix));CHKERRQ(ierr);
  PetscReal (*Stoichiometry_matrix)[2] = (PetscReal (*)[2])  (user->stoichiometry_matrix);

  Stoichiometry_matrix[0][0] = -1.0; Stoichiometry_matrix[0][1] = 1.0;
  Stoichiometry_matrix[1][0] = -1.0; Stoichiometry_matrix[1][1] = 1.0;
  Stoichiometry_matrix[2][0] =  2.0; Stoichiometry_matrix[2][1] = -2.0;
  
  PetscPrintf(PETSC_COMM_WORLD,"Stoichiometry matrix:  \n");
  for(i=0; i<user->nf; i++){
    for(j=0; j<2; j++){
      PetscPrintf(PETSC_COMM_WORLD,"stoichiometry_matrix[%d][%d]= %1.0f ",i,j,Stoichiometry_matrix[i][j]);
    }
    PetscPrintf(PETSC_COMM_WORLD,"\n");
  }
  PetscPrintf(PETSC_COMM_WORLD,"\n");

  PetscFunctionReturn(0);
}

#undef  __FUNCT__
#define __FUNCT__ "IniCondSystem_4"
PetscErrorCode IniCondSystem_4(IGAPoint p,PetscScalar *F,void *ctx)
{
  AppCtx *user = (AppCtx *)ctx;

  PetscInt a;
  PetscInt nen    = p->nen;
  PetscInt dof    = p->dof;
  PetscReal pos[p->dim];
  IGAPointFormPoint(p,pos);

  IGAProbe Unit = user->UAC;
  PetscScalar u0;
  IGAProbeSetPoint(Unit,p->point);
  IGAProbeFormValue(Unit,&u0);

  PetscReal Phi0, Phi1;

  if(u0 > 0.8){
    Phi0 = 0.78;
    Phi1 = 0.21;
  }
  else{
    Phi0 = 0.21;
    Phi1 = 0.78;
  }

  const PetscReal *N0 = (typeof(N0)) p->shape[0];
  PetscScalar (*Fa)[dof]           = (PetscScalar (*)[dof])F;
  for(a=0; a<nen; a++) {
    Fa[a][0] = N0[a]*Phi0;
    Fa[a][1] = N0[a]*Phi1;
    Fa[a][2] = N0[a]*0.0;
    Fa[a][3] = N0[a]*0.0;
  }

  return 0;
}

#undef __FUNCT__
#define __FUNCT__ "MassVector"
PetscErrorCode MassVector(IGAPoint p, PetscScalar *V, void *ctx){

  PetscInt a, nen = p->nen;
  const PetscReal *N0;
  IGAPointGetShapeFuns(p,0,(const PetscReal**)&N0);

  PetscScalar (*Ve)[4] = (PetscScalar (*)[4])V;

  for(a=0;a<nen;a++){
    Ve[a][0] = N0[a];
    Ve[a][1] = N0[a];
    Ve[a][2] = N0[a];
    Ve[a][3] = N0[a];
  }
  return 0;
}

#undef  __FUNCT__
#define __FUNCT__ "FreeEnergy"
PetscErrorCode FreeEnergy(IGAPoint p,const PetscScalar *U,PetscInt n,PetscScalar *S, void *ctx)
{
  PetscErrorCode  ierr;
  AppCtx *user = (AppCtx *)ctx;

  PetscFunctionBegin;

  PetscInt a,b,i,c,j;
  PetscInt dim = p->dim;
  PetscInt nf  = user->nf;
  PetscScalar beta = user->factor;

  PetscScalar u[nf-1+dim];
  PetscScalar du[nf-1+dim][dim];
  IGAPointFormValue(p,U,&u[0]);
  IGAPointFormGrad(p,U,&du[0][0]);

  IGAProbe Unit = user->Unit;
  PetscScalar u0[nf-1+dim];
  PetscScalar du0[nf-1+dim][dim];
  
  IGAProbeSetPoint(Unit,p->point);
  IGAProbeFormValue(Unit,&u0[0]);
  IGAProbeFormGrad(Unit,&du0[0][0]);
  
  PetscScalar phi0[nf], dphi0[nf][dim];
  PetscScalar phi[nf], dphi[nf][dim];
  
  for(c=0;c<nf-1;c++){ // component
    phi0[c] = u0[c];
    phi[c] = u[c];
    for(i=0;i<dim;i++){  //gradient
      dphi0[c][i] = du0[c][i];
      dphi[c][i] = du[c][i];
    }
  }
  
  PetscScalar d2u[nf-1+dim][dim][dim];
  PetscScalar d2phi[nf][dim][dim];
  IGAPointFormHess (p,U,&d2u[0][0][0]);

  // Splitting the Hessian for phases
  for(c=0;c<nf-1;c++){
    for(i=0;i<dim;i++){
      for(j=0;j<dim;j++){
	d2phi[c][i][j] = d2u[c][i][j];
      }
    }
  }

  PetscScalar delphi[nf];
  //Gather the laplacians
  for(c=0; c<nf-1; c++){   
    delphi[c] = 0.0;
    for(i=0; i<dim; i++){
      delphi[c] += d2phi[c][i][i];
    }
  }
  
  //Applying Lattice constrain $sum_{a=1}^{nf} = 1.0$
  phi0[nf-1] = 1.0;
  phi[nf-1] = 1.0;
  delphi[nf-1] = 0.0;
  for(i=0; i<dim; i++){ dphi0[nf-1][i] = 0.0; dphi[nf-1][i] = 0.0;}

  //Computing the component nf
  for(c=0; c<nf-1; c++){ 
      phi0[nf-1] += -phi0[c];
      phi[nf-1] += -phi[c];
      delphi[nf-1] += -delphi[c];
      for(i=0; i<dim; i++){ dphi0[nf-1][i] += -dphi0[c][i]; dphi[nf-1][i] += -dphi[c][i];}
  }
  
  PetscScalar dd[dim][dim];
  for(i=0;i<dim;i++){
    for(j=0;j<dim;j++){
      dd[i][j] = du[nf-1+i][j];
    }
  }

    
  PetscScalar F[dim][dim];
  PetscScalar InvF[dim][dim];
  PetscScalar J;
    
  ierr = TensorOperations(dim, &J, &dd[0][0], &F[0][0], &InvF[0][0],user);CHKERRQ(ierr);

  PetscScalar VolSweSum = 0.0;
  for (i=0;i<user->nf;i++){VolSweSum += user->omega_phi[i]*(phi[i]-phi0[i]);}
  PetscScalar Jphi = 1.0 + VolSweSum;

  PetscScalar FF=0.0;
  for(i=0;i<dim;i++){
    for(j=0;j<dim;j++){
      FF += F[i][j]*F[i][j];
    }
  }

  PetscScalar elastic_energy = user->Psi0*user->e*(0.5*(FF*pow(Jphi,((PetscScalar)-2/3))-3.0) + (1.0/beta)*(pow(pow(Jphi,((PetscScalar)-2/3))*J,-beta)-1.0));
  
  const PetscReal (*theta_c)[nf] = (typeof(theta_c)) user->theta_c;
  const PetscReal (*cahn)[nf] = (typeof(cahn)) user->cahn;
  const PetscReal (*lsigma)[nf] = (typeof(lsigma)) user->lsigma;
  
  PetscReal sum1=0.0, sum2=0.0, sum3=0.0;
  for (a=0; a < nf; a++){
    //bulk
    sum1 += phi[a]*log(phi[a]);
    for (b=0; b < nf; b++){ 
      sum2 += theta_c[a][b]*phi[a]*phi[b];
      sum3 += 0.5*(lsigma[a][b]*cahn[a][b])*DOT(dim,dphi[a],dphi[b]);   //interface
    }
  }
  
  //Total free energy
  S[0] = user->Psi0*(0.5*sum1 + sum2 + sum3); 

  //Interfacial energy
  S[1] = user->Psi0*0.5*(lsigma[0][0]*cahn[0][0])*DOT(dim,dphi[0],dphi[0]);
  S[2] = user->Psi0*0.5*(lsigma[1][1]*cahn[1][1])*DOT(dim,dphi[1],dphi[1]);
  S[3] = user->Psi0*0.5*(lsigma[2][2]*cahn[2][2])*DOT(dim,dphi[2],dphi[2]);
  
  //Mass
  S[4] = phi[0];
  S[5] = phi[1];
  S[6] = phi[2];

  //Elastic Energy

  S[7] = elastic_energy;
  
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "EnergyMonitor"
PetscErrorCode EnergyMonitor(TS ts,PetscInt step,PetscReal t,Vec U,void *mctx)
{ 
  AppCtx *user = (AppCtx *)mctx;
  PetscErrorCode ierr;
  PetscFunctionBegin;

  IGA iga;
  ierr = PetscObjectQuery((PetscObject)ts,"IGA",(PetscObject*)&iga);CHKERRQ(ierr);

  SNES        snes;
  PetscScalar dt;
  PetscInt    snes_it,ksp_it;
  ierr = TSGetTimeStep(ts,&dt);CHKERRQ(ierr);
  ierr = TSGetSNES(ts,&snes);CHKERRQ(ierr);
  ierr = SNESGetIterationNumber(snes,&snes_it);CHKERRQ(ierr);
  ierr = SNESGetLinearSolveIterations(snes,&ksp_it);CHKERRQ(ierr);
  
  PetscScalar result[8];
  ierr = IGAComputeScalar(iga,U,8,result,FreeEnergy,mctx);CHKERRQ(ierr);
  PetscScalar energy = result[0];
  PetscScalar i11 = result[1];
  PetscScalar i22 = result[2];
  PetscScalar i33 = result[3];
  PetscScalar m1   = result[4];
  PetscScalar m2   = result[5];
  PetscScalar m3   = result[6];
  PetscScalar elastic = result[7];
  
  if(snes_it > 0) ksp_it /= snes_it;

  if (step == 0) {
    PetscPrintf(PETSC_COMM_WORLD,"Time            FreeEnergy     Inter_12 Mass_1\n");
  }
  if(step > 0 && energy > user->energy){
    PetscPrintf(PETSC_COMM_WORLD,"%d %.6e %.16e %.16e %.16e %.16e %.16e %.16e %.16e %.16e\n",step,t,energy,i11,i22,i33,m1,m2,m3,elastic);
  }else{
    PetscPrintf(PETSC_COMM_WORLD,"%d %.6e %.16e %.16e %.16e %.16e %.16e %.16e %.16e %.16e\n",step,t,energy,i11,i22,i33,m1,m2,m3,elastic);
  }

  user->energy = energy;
  
  PetscFunctionReturn(0);
}

#undef __FUNCT__
#define __FUNCT__ "main"
int main(int argc, char *argv[]) {

  PetscErrorCode  ierr;
  ierr = PetscInitialize(&argc,&argv,0,0);CHKERRQ(ierr);

  AppCtx user;
  ierr = Init3PhasesParameters(&user);CHKERRQ(ierr);

  PetscInt  dim     = 2;
  PetscInt  N       = 64;
  PetscInt  p       = 2;
  PetscInt  k       = 1;
  //char initial[PETSC_MAX_PATH_LEN] = {0};
  PetscBool output  = PETSC_FALSE;
  PetscBool monitor = PETSC_FALSE;
  //Loading initial condition
  PetscInt  init    = 0;
  PetscBool load    = PETSC_FALSE;
  //PetscInt       time_steps_max = 100;
  //PetscReal      time_total_max = 100.0;
  //PetscReal time_step = 1e-9;
  
  ierr = PetscOptionsBegin(PETSC_COMM_WORLD,"","Cahn Hilliard Options","IGA");CHKERRQ(ierr);
  ierr = PetscOptionsInt("-N","number of elements",__FILE__,N,&N,PETSC_NULL);CHKERRQ(ierr);
  ierr = PetscOptionsInt("-p","polynomial order",  __FILE__,p,&p,PETSC_NULL);CHKERRQ(ierr);
  ierr = PetscOptionsInt("-k","continuity order",  __FILE__,k,&k,PETSC_NULL);CHKERRQ(ierr);
  //ierr = PetscOptionsString("-initial","Load initial solution from file",__FILE__,initial,initial,sizeof(initial),NULL);CHKERRQ(ierr);
  ierr = PetscOptionsInt("-init"   ,"Initial iteration"               ,__FILE__,init        ,&init    ,PETSC_NULL);CHKERRQ(ierr);
  ierr = PetscOptionsBool("-load"  ,"Load initial condition"          ,__FILE__,load        ,&load    ,PETSC_NULL);CHKERRQ(ierr);
  ierr = PetscOptionsBool("-output","Enable output files"             ,__FILE__,output      ,&output  ,PETSC_NULL);CHKERRQ(ierr);
  ierr = PetscOptionsBool("-monitor","Monitor solution details"       ,__FILE__,monitor     ,&monitor ,PETSC_NULL);CHKERRQ(ierr);
  ierr = PetscOptionsReal("-ch_C","Characteristic parameter"          ,__FILE__,user.C      ,&user.C,NULL);CHKERRQ(ierr);
  ierr = PetscOptionsReal("-ch_phibar","Initial average concentration",__FILE__,user.phibar ,&user.phibar,NULL);CHKERRQ(ierr);
  ierr = PetscOptionsReal("-ch_eps","Interfacial parameter"           ,__FILE__,user.eps    ,&user.eps,NULL);CHKERRQ(ierr);
  ierr = PetscOptionsEnd();CHKERRQ(ierr);
  
  IGA      iga;
  ierr = IGACreate(PETSC_COMM_WORLD,&iga);CHKERRQ(ierr);
  ierr = IGASetDim(iga,dim);CHKERRQ(ierr);
  ierr = IGASetDof(iga,user.nf-1+dim);CHKERRQ(ierr);
  
  PetscInt i;
  IGAAxis  axis;
  for(i=0;i<dim;i++){
    ierr = IGAGetAxis(iga,i,&axis);CHKERRQ(ierr);
    ierr = IGAAxisSetPeriodic(axis,PETSC_TRUE);CHKERRQ(ierr);
    ierr = IGAAxisSetDegree(axis,p);CHKERRQ(ierr);
    ierr = IGAAxisInitUniform(axis,N,0.0,1.0,k);CHKERRQ(ierr);
  }
  ierr = IGASetFromOptions(iga);CHKERRQ(ierr);
  ierr = IGASetUp(iga);CHKERRQ(ierr);
  ierr = IGAWrite(iga,"iga.dat");CHKERRQ(ierr);
  
  TS ts;
  ierr = IGACreateTS(iga,&ts);CHKERRQ(ierr);
  ierr = TSSetTimeStep(ts,1e-11);CHKERRQ(ierr);
  ierr = TSSetType(ts,TSALPHA1);CHKERRQ(ierr);
  ierr = TSAlphaSetRadius(ts,0.5);CHKERRQ(ierr);
  ierr = TSAlphaUseAdapt(ts,PETSC_TRUE);CHKERRQ(ierr);
  ierr = TSSetMaxSNESFailures(ts,-1);CHKERRQ(ierr);
  
  ierr = IGASetFormIFunction(iga,ResidualPrimal,&user);CHKERRQ(ierr);
  ierr = IGASetFormIJacobian(iga,IGAFormIJacobianFD,&user);CHKERRQ(ierr);
  
  if (monitor) {ierr = TSMonitorSet(ts,EnergyMonitor,&user,PETSC_NULL);CHKERRQ(ierr);}
  if (output)  {
    // OutputMonitor parameters
    if (load) {user.init = init;}
    else      {user.init = 0;}
    //ierr = TSMonitorSet(ts,OutputMonitor,&monit,NULL);CHKERRQ(ierr);
    ierr = TSMonitorSet(ts,OutputMonitor,&user,PETSC_NULL);CHKERRQ(ierr);
  }
  ierr = TSSetFromOptions(ts);CHKERRQ(ierr);
  
  Vec U;
  ierr = IGACreateVec(iga,&U);CHKERRQ(ierr);
  //if (initial[0] != 0) { /* initial condition from datafile */
  if (load){
    //ierr = IGAReadVec(iga,U,initial);CHKERRQ(ierr);
    char initname[256];
    sprintf(initname,"./ch-%d.dat",init);CHKERRQ(ierr);
    ierr = IGAReadVec(iga,U,initname);CHKERRQ(ierr);
  } 
  else {                /* initial condition is random */
        //ierr = FormInitialCondition(iga,U,&user);CHKERRQ(ierr);
    /*
    Mat K;
    KSP kspic;
    PC  pcic;
    ierr = IGACreateVec(iga,&U);CHKERRQ(ierr);
    ierr = IGACreateMat(iga,&K);CHKERRQ(ierr);
    ierr = IGASetFormSystem(iga,IniCondSystem,&user);CHKERRQ(ierr); 
    ierr = IGAComputeSystem(iga,K,U);CHKERRQ(ierr);
    ierr = IGACreateKSP(iga,&kspic);
    ierr = KSPSetOperators(kspic,K,K);CHKERRQ(ierr);
    ierr = KSPSetOptionsPrefix(kspic,"IC_");CHKERRQ(ierr);
    ierr = KSPGetPC(kspic, &pcic);CHKERRQ(ierr);
    //ierr = PCSetType(pcic, PCLU);CHKERRQ(ierr);
    //ierr = KSPSetTolerances(kspic,1e-6,1e-8,1e-10,10000);CHKERRQ(ierr);
    ierr = KSPSetFromOptions(kspic);CHKERRQ(ierr);
    ierr = KSPSolve(kspic,U,U);CHKERRQ(ierr);
    ierr = MatDestroy(&K);CHKERRQ(ierr);
    */

    char name[256];
    Vec UinitAC;
    IGA igainit;

    ierr = IGACreate(PETSC_COMM_WORLD,&igainit);CHKERRQ(ierr);
    ierr = IGARead(igainit,"igainit.dat");CHKERRQ(ierr);
    ierr = IGASetDof(igainit,1);CHKERRQ(ierr);
    ierr = IGASetUp(igainit);CHKERRQ(ierr);

    ierr = IGACreateVec(igainit,&UinitAC);CHKERRQ(ierr);
    sprintf(name,"./init%d.dat",125);
    ierr = IGAReadVec(igainit,UinitAC,name);CHKERRQ(ierr);

    IGAProbe UAC;
    ierr = IGAProbeCreate(igainit,UinitAC,&UAC);CHKERRQ(ierr);
    ierr = IGAProbeSetCollective(UAC,PETSC_FALSE);CHKERRQ(ierr);
    ierr = IGAProbeSetOrder(UAC,1);CHKERRQ(ierr);
    user.UAC = UAC;

    Vec Q,f;
    ierr = IGACreateVec(iga, &Q);CHKERRQ(ierr);
    ierr = IGACreateVec(iga, &f);CHKERRQ(ierr);
    ierr = IGASetFormVector(iga,IniCondSystem_4,&user);CHKERRQ(ierr);
    ierr = IGAComputeVector(iga,f);CHKERRQ(ierr);
    ierr = IGASetFormVector(iga,MassVector,NULL);CHKERRQ(ierr);
    ierr = IGAComputeVector(iga,Q);CHKERRQ(ierr);
    ierr = VecPointwiseDivide(U, f, Q);CHKERRQ(ierr);
    
    }
  
  /*
  IGA IGAGather;
  ierr = IGAClone(iga,1,&IGAGather);CHKERRQ(ierr);
  ierr = IGASetUp(IGAGather);CHKERRQ(ierr);
  Vec UnitGather;
  ierr = IGACreateVec(IGAGather,&UnitGather);CHKERRQ(ierr);
  ierr = VecStrideGather(U,0,UnitGather,INSERT_VALUES);CHKERRQ(ierr);
  */  

  PetscScalar zero = 0.0;
  IGA IGAScatter;
  ierr = IGAClone(iga,1,&IGAScatter);CHKERRQ(ierr);
  ierr = IGASetUp(IGAScatter);CHKERRQ(ierr);
  Vec UnitScatter;
  ierr = IGACreateVec(IGAScatter,&UnitScatter);CHKERRQ(ierr); 
  ierr = VecSet(UnitScatter,zero);

  // ierr = VecStrideScatter(UnitScatter,2,U,INSERT_VALUES);CHKERRQ(ierr);
  // ierr = VecStrideScatter(UnitScatter,3,U,INSERT_VALUES);CHKERRQ(ierr);
  /*
  ierr = PetscViewerDrawSetPause(PETSC_VIEWER_DRAW_WORLD,-1);CHKERRQ(ierr);
  ierr = VecView(U,PETSC_VIEWER_DRAW_WORLD);CHKERRQ(ierr);
  */
  IGAProbe Unit;
  ierr = IGAProbeCreate(iga,U,&Unit);CHKERRQ(ierr);
  ierr = IGAProbeSetCollective(Unit,PETSC_FALSE);CHKERRQ(ierr);
  ierr = IGAProbeSetOrder(Unit,1);CHKERRQ(ierr);
  user.Unit = Unit;
  
  ierr = TSSolve(ts,U);CHKERRQ(ierr);
  ierr = IGAWriteVec(iga,U,"ch-last.dat");CHKERRQ(ierr);

  ierr = VecDestroy(&U);CHKERRQ(ierr);
  ierr = TSDestroy(&ts);CHKERRQ(ierr);
  ierr = IGADestroy(&iga);CHKERRQ(ierr);
  ierr = PetscFinalize();CHKERRQ(ierr);
  
  return 0;
}
